package org.jcon.core.tron;

/**
* This interface contains convenience methods for use in
* collections that have String keys and Object values.
* This improves understandability and reduces code bloat.
*
* @author Jack Harich
*/

public interface ConvenientStringMap { 

// ----- String methods
/**
* Sets the key value, overwriting any that already exists.
* @param key   the unique key, which may not be null.
* @param value the String value, which may be null.
*/
public void setString(String key, String value);

/**
* Returns the String value  for the key.
* @param key  the key to be used for the lookup.
* @return     the key's value or null.
*/
public String getString(String key);

/**
* Same as getString(key) except allows a default to be
* returned rather than null.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if the
*    return would otherwise be null.
* @return     the key's value or the defaultValue if not found.
*/
public String getStringDefault(String key, String defaultValue);

// ----- boolean methods
/**
* Sets the key value, overwriting any that already exists.
* @param key   the unique key, which may not be null.
* @param value the boolean value.
*/
public void setBoolean(String key, boolean value);

/**
* Same as setBoolean(key, true).
*/
public void setTrue(String key);

/**
* Same as setBoolean(key, false).
*/
public void setFalse(String key);

/**
* Returns the boolean value for the key.
* @param key  the key to be used for the lookup.
* @return     the key's value or an exception if not found.
*/
public boolean getBoolean(String key); // For symmetry

/*
* Used to determine if the key value is true.
* @param key  the key in question.
* @return     true if the key's value is true, false if
*    it's false, or an exception if not found.
*/
public boolean isTrue(String key);

/**
* Same as isTrue(key) except allows a default to be
* returned if not found rather than an exception.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if the
*    value is not found.
* @return     true if the key's value is true, false if
*    it's false, or the defaultValue if not found.
*/
public boolean isTrueDefault(String key, boolean defaultValue);

/*
* Used to determine if the key value is false.
* @param key  the key in question.
* @return     true if the key's value is false, false if
*    it's true, or an exception if not found.
*/
public boolean isFalse(String key);

/**
* Same as isFalse(key) except allows a default to be
* returned if not found rather than an exception.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if the
*    value is not found.
* @return     true if the key's value is false, false if
*    it's true, or the defaultValue if not found.
*/
public boolean isFalseDefault(String key, boolean defaultValue);

// ----- int methods
/**
* Sets the key value, overwriting any that already exists.
* @param key   the unique key, which may not be null.
* @param value the int value.
*/
public void setInt(String key, int value);

/**
* Returns the int value for the key.
* @param key  the key to be used for the lookup.
* @return     the key's value or an exception if not found.
*/
public int getInt(String key);

/**
* Same as getInt(key) except allows a default to be
* returned rather than an exception if not found.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if not found.
* @return the key's value or the defaultValue if not found.
*/
public int getIntDefault(String key, int defaultValue);

// ----- long methods
/**
* Sets the key value, overwriting any that already exists.
* @param key   the unique key, which may not be null.
* @param value the long value.
*/
public void setLong(String key, long value);

/**
* Returns the long value for the key.
* @param key  the key to be used for the lookup.
* @return     the key's value or an exception if not found.
*/
public long getLong(String key);

/**
* Same as getLong(key) except allows a default to be
* returned rather than an exception if not found.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if not found.
* @return the key's value or the defaultValue if not found.
*/
public long getLongDefault(String key, long defaultValue);

// ----- float methods
/**
* Sets the key value, overwriting any that already exists.
* @param key   the unique key, which may not be null.
* @param value the float value.
*/
public void setFloat(String key, float value);

/**
* Returns the float value for the key.
* @param key  the key to be used for the lookup.
* @return     the key's value or an exception if not found.
*/
public float getFloat(String key);

/**
* Same as getFloat(key) except allows a default to be
* returned rather than an exception if not found.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if not found.
* @return the key's value or the defaultValue if not found.
*/
public float getFloatDefault(String key, float defaultValue);

// ----- double methods
/**
* Sets the key value, overwriting any that already exists.
* @param key   the unique key, which may not be null.
* @param value the double value.
*/
public void setDouble(String key, double value);

/**
* Returns the double value for the key.
* @param key  the key to be used for the lookup.
* @return     the key's value or an exception if not found.
*/
public double getDouble(String key);

/**
* Same as getDouble(key) except allows a default to be
* returned rather than an exception if not found.
* @param key  the key to be used for the lookup.
* @param defaultValue the default to be used if not found.
* @return the key's value or the defaultValue if not found.
*/
public double getDoubleDefault(String key, double defaultValue);

}