package org.jcon.core.proto.c2;

/**
* This interface represents a single node in a hierarchial
* System Tree. Each node has one part. A node can be a 
* container or leaf. Containers hold zero or more nodes.
* The terms part and node are fairly interchangable.
* <p>
* Responsibilites are to maintain the state that applies
* to all parts, such as name, parent and isCore. The design
* philosophy is it should remain small and simple, serving
* only an "identification and properties role", with no 
* behavior.
* <p>
* The boolean properties are all designed with the most 
* common case as false, which is the default. For example
* we use "isHidden" instead of "isVisible".
* <p>
* TODO - A design problem is how do we add more properties,
* or hide existing properties? Perhaps we need a propMap.
*
* @author Jack Harich
*/
public interface Node {

//---------- name ----------------------------------------
/**
* Sets the node or part name. This must be unique per container.
* It must be alphanumeric, may contain periods, and must
* not contain blanks or slashes. It cannot be null or a
* zero length String. This is a required property.
* <p>
* Our convention is to capatalize the first letter of each
* word making up the self-documenting part name. Examples
* are "Root", "ControlRegistry" and "UserSecurity".
* <p>
* If part names are used for finding the part in other
* containers, then to avoid name space conflicts the names
* need careful design. This can achieve branch or system
* uniqueness. This will happen if service parts are used
* a lot. To support unique naming we allow periods in the
* name, such as "org.jcon.security.UserAuthorization".
*/
public void setName(String name);

/*
* Returns the part name.
*/
public String getName();

//---------- part ----------------------------------------
/**
* Sets the node's part, which may be any object whatsoever.
* The object should be unique per container and probably
* unique per System Tree. This is a required property.
* <p>
* If the part implements NodeUser then setNode(Node) will
* be called, using "this" node.
*/
public void setPart(Object part);

/**
* Returns the node's part.
*/
public Object getPart();

//---------- paremt --------------------------------------
/*
* Sets the node's parent node. This is required for 
* all nodes in a tree except the root, which is null.
*/
public void setParentNode(Node parentNode);

/*
* Returns the node's parent node or null if this is
* the root.
*/
public Node getParentNode();

//---------- isCore --------------------------------------
/**
* Sets whether this is a core part or not. The default is
* false.
*/
public void setCore(boolean isCore);

/**
* Returns true if this is a core part or false if not.
* <p>
* Parts are divided into two groups:
* core and domain. Core parts are required for a System 
* Tree to run. They are domain neutral, and are normally 
* hidden. Domain parts are the ones supplied by the system
* builder, have  the domain functionality, and form the 
* bulk of a system. The use of core parts by domain parts
* should be minimized to preserve core encapsulation.
*/
public boolean isCore();

//---------- isContainer ---------------------------------
/**
* Sets whether this is a container part or not. The default
* is false.

*/
public void setContainer(boolean isContainer);

/**
* Returns true if this is a container part or false if not.
* <p>
* A unique type of core part
* is a  container. It holds other container or leaf parts.
* All other types of parts are leaf parts. A container part
* requires the other core parts to run. Thus the core parts
* form the "container" portion of a system. We do not call
* this "isCell" to allow other types of containers, such as
* organs.
*/
public boolean isContainer();

//---------- isCompanion ---------------------------------
/**
* Sets whether this is a companion part or not. The default
* is false.
*/
public void setCompanion(boolean isCompanion);

/**
* Returns true if this is a companion part or false if not.
* <p>
* A container may
* have one leaf part that's psychologically associated with
* the container. An example is a registry. In the visual 
* System Tree companions are shown as with their containers
* instead of being a child. This helps conceptual design.
*/
public boolean isCompanion();


//---------- isInheritable -------------------------------
/**
* Sets whether this is inheritable or not. The default is
* false.
*/
public void setInheritable(boolean isInheritable);

/**
* Returns true if this is inheritable or false if not.
* <p>
* Parts may play a "local"
* or "inheritable" role in their collaborations with other
* parts. Most parts are local, and are used via Messages or 
* occasionally directly. Some parts play a "service" role
* by sitting near the upper region of a System Tree and 
* serving the needs of many parts below by being inheritable.
* This is the Service Architecture Pattern, though that
* pattern is not limited to trees.
* <p>
* We recommend the use of Messages and MessageService 
* instead of service parts for most use. This has looser
* coupling.
* <p>
* An example of the use of isInheritable is core service
* parts. This allows cores that don't use Messages,
* gives greater speed, and allows part sharing of stateless
* part by many containers for efficiency. One purpose
* of this property is so the designer can declare that a
* part NOT be used as a service, which can improve cell
* encapsulation. Another advantage is automatic management
* of service parts by a service registry or such.
*/
public boolean isInheritable();

//---------- isVirtual -----------------------------------
/**
* Sets whether this is a virtual part or not. The default
* is false.
*/
public void setVirtual(boolean isVirtual);

/**
* Returns true if this is a virtual part or false if not.
* <p>
* Parts may be normal or 
* "virtual". Normally all parts are defined in Container 
* DK. But virtual parts are not defined by Container DK. 
* This  allows more dynamic system behavior, less DK, and 
* lighter system weight. An example of virtual parts is 
* core parts below the root. 
* <p>
* TODO - Perhaps we need a better word, 
* but "virtual" means a part definition is hidden, and so
* the part appears out of nowhere. Is this property really
* needed?
*/
public boolean isVirtual();

//---------- isHidden -----------------------------------
/**
* Sets whether this is a hidden part or not. The default
* is false.
*/
public void setHidden(boolean isHidden);

/**
* Returns true if this is a hidden part or false if not.
* <p>
* Most parts are visible in the
* visual System Tree tool and to "get all parts in a 
* container" calls, etc. But the core parts (except for 
* container) are hidden, because that would clutter up 
* things visually, and expose them to possible misuse by
* non-core parts. This also helps preserve core 
* encapsulation.
* <p>
* This property can be used to control the behavior of
* classes like "PartReader" to hide sensitive parts.
* (It's a little muddy because of the show container
* exception)
*/
public boolean isHidden();

//

//----- Other
/**
* Returns the root node, which has no parent node.
*/
public Node getRootNode();

/**
* Returns a String representation of the node. The standard
* is the name.
*/
public String toString();



} // End interface