package org.jcon.core.proto.c1;

/**
* This interface represents a single part in a node in a 
* System Tree. Each node has one part. A part can be a 
* container or leaf. Containers hold zero or more parts.
* <p>
* Responsibilites are to maintain the state that applies
* to all parts, such as name, parent and isCore. The design
* philosophy is it should remain small and simple, serving
* only an "identification and properties role", with no 
* behavior.
* <p>
* We now discuss the various types of parts, which lead to
* the properties a part can have. 
* <p>
* The boolean properties are all designed with the most 
* common case as false, which is the default. For example
* we use "isHidden" instead of "isVisible".
* <p>
* Core Parts (isCore) - Parts are divided into two groups:
* core and domain. Core parts are required for a System 
* Tree to run. They are domain neutral, and are normally 
* hidden. Domain parts are the ones supplied by the system
* builder, have  the domain functionality, and form the 
* bulk of a system. In general, core parts should only be 
* used by other core parts, to preserve core encapsulation.
* <p>
* Container Part (isContainer) - A unique type of core part
* is a  container. It holds other container or leaf parts. All
* other types of parts are leaf parts. A container part
* requires the other core parts to run. Thus the core parts
* form the "container" portion of a system. We do not call
* this "isCell" to allow other types of containers, such as
* organs.
* <p>
* Container Companion Part (isCompanion) - A container may
* have one leaf part that's psychologically associated with
* the container. An example is a registry. In the visual 
* System Tree companions are shown as with their containers
* instead of being a child. This helps conceptual design.
* <p>
* Service Parts (isService) - Parts may play a "local" or
* "service" role in their collaborations with other parts.
* Most parts are local, and are used via Messages or 
* occasionally directly. Some parts play a service role by
* sitting near the upper region of a System Tree and 
* serving the needs of many parts below. This is usually
* done by the part being a "Message Service". An exception
* is core service parts, which are used directly for speed
* and shared by many containers for efficiency. One purpose
* of this property is so the designer can declare that a
* part NOT be used as a service, which can improve cell
* encapsulation. Another advantage is automatic management
* of service parts by a service registry or such.
* <p>
* PROBLEM - Can't any part be a service, ie be used by
* others below? This property was created to solve the
* need for not duplicating core stateless parts. Could add
* stateless property??? This would clarity and encourage
* reuse. - Rewrite to use philosophy that stateless parts
* allow a certain type of reuse. Replace isService with
* isStateless. - OPEN DECISION
* <p>
* Virtual Parts (isVirtual) - Parts may be normal or 
* "virtual". Normally all parts are defined in Container 
* DK. But virtual parts are not defined by Container DK. 
* This  allows more dynamic system behavior, less DK, and 
* lighter system weight. An example of virtual parts is 
* core parts below the root.
* <p>
* Hidden Parts (isHidden) - Most parts are visible in the
* visual System Tree tool and to "get all parts in a 
* container" calls, etc. But the core parts (except for 
* container) are hidden, because that would clutter up 
* things visually, and expose them to possible misuse by
* non-core parts. This also helps preserve core 
* encapsulation.
*
* @author Jack Harich
*/
public interface Part extends Cloneable {

//---------- name ----------------------------------------
/**
* Sets the part name. This must be unique per cell.
* It must be alphanumeric, may contain periods, and must
* not contain blanks or slashes. It cannot be null or a
* zero length String. This is a required property.
* <p>
* Our convention is to capatalize the first letter of each
* word making up the self-documenting part name. Examples
* are Root, ControlRegistry and UserSecurity.
* <p>
* If part names are used for finding the part in other
* containers, then to avoid name space conflicts the names
* need careful design. This can achieve branch or system
* uniqueness. This will happen if service parts are used
* a lot. To support unique naming we allow periods in the
* name, such as org.jcon.security.UserAuthorization.
*/
public void setName(String name);

/*
* Returns the part name.
*/
public String getName();

//---------- paremt --------------------------------------
/*
* Sets the part's parent cell. This is required for 
* all parts in a tree except the root, which is null.
*/
public void setParent(Cell parent);

/*
* Returns the part's parent cell or null if this is
* the root.
*/
public Cell getParent();

//---------- isCore --------------------------------------
/**
* Sets whether this is a core part or not. The default is
* false.
*/
public void setCore(boolean isCore);

/**
* Returns true if this is a core part or false if not.
*/
public boolean isCore();

//---------- isContainer ---------------------------------
/**
* Sets whether this is a container part or not. The default
* is false.

*/
public void setContainer(boolean isContainer);

/**
* Returns true if this is a container part or false if not.
*/
public boolean isContainer();

//---------- isCompanion ---------------------------------
/**
* Sets whether this is a companion part or not. The default
* is false.
*/
public void setCompanion(boolean isCompanion);

/**
* Returns true if this is a companion part or false if not.
*/
public boolean isCompanion();


//---------- isService -----------------------------------
/**
* Sets whether this is a service part or not. The default is
* false.
*/
public void setService(boolean isService);

/**
* Returns true if this is a service part or false if not.
*/
public boolean isService();

//---------- isVirtual -----------------------------------
/**
* Sets whether this is a virtual part or not. The default is
* false.
*/
public void setVirtual(boolean isVirtual);

/**
* Returns true if this is a virtual part or false if not.
*/
public boolean isVirtual();

//---------- isHidden -----------------------------------
/**
* Sets whether this is a hidden part or not. The default is
* false.
*/
public void setHidden(boolean isHidden);

/**
* Returns true if this is a hidden part or false if not.
*/
public boolean isHidden();

//----- Other
/**
* Returns a shallow clone of the worker.
*/
public Object clone() throws CloneNotSupportedException;


} // End interface