package org.jcon.core.part.std;

import org.jcon.core.part.Cell;
import org.jcon.core.part.Node;
import org.jcon.core.part.PartReader;
import java.util.Enumeration;
import java.util.Vector;

/**
* This class allows reading parts from a cell. It acts
* as a cell Fascade. (More methods will be added)
*
* @author Jack Harich
*/
public class PartReaderStd implements PartReader {

//---------- Protected Fields ----------------------------
protected Cell myCell;

//---------- PartReader Implementation -------------------
/**
* Finds and returns the named part. 
* See <a href="org.jcon.core.part.PartReader.html#findPart(java.lang.String)">full documentation</a>.
*
* @param partName  the name of the part to find.
* @return          the part found.
*/
public Object findPart(String partName) {
    return myCell.findPart(partName);
}
/**
* Returns the named part in the cell.
* See <a href="org.jcon.core.part.PartReader.html#getPart(java.lang.String)">full documentation</a>.
*
* @param name  the name of the part to retrieve.
* @return      the part in the cell with that name.
*/
public Object getPart(String name) {
    return myCell.getPart(name);
}
/**
* Returns an Enumeration of the part names in the cell, in order. 
* @return  the part names in the cell.
*/
public Enumeration enumeratePartNames() {
    return myCell.enumerateNodeNames();
}
/**
* Returns the number of parts in the cell.
* @return  the number of parts in the cell.
*/
public int getPartCount() {
    return myCell.getNodeCount();
}
/**
* Returns a String representing the System Tree, starting
* at the root or at the cell this PartReader is in.
* Format is one line per part with indentation. This is for 
* simple developement use.
*
* @param startAtRoot  true to start at the root or false to
*                     start at this cell.
*
* @return  a String representing the System Tree.
*/
public String listTree(boolean startAtRoot) {
    Cell startingCell = myCell;
    
    if (startAtRoot) {
        startingCell = (Cell)myCell.getCellNode().getRootNode().getPart();
    }
    String text = startingCell.getCellNode().getName();
    return text + "\n" + listCell(startingCell, 1);    
}
//----- PartUser Implementation 
/**
* Returns an array of the part names this part needs to
* do its work.
* @return  the names of the parts needed.
*/
public String[] getPartNamesNeeded() {
    return new String[] {"MyCell"};
}
/**
* Sets a named part, in response to part needs.
* @param name  the name of the needed part.
* @param part  the needed part.
*/
public void setPart(String name, Object part) {
    if (name.equals("MyCell")) myCell = (Cell)part;
}
//------ Replicatable Implementation 
/**
* Returns a copy of itself without mission state.
* @return  the replication.
*/
public Object replicate() {
    // No DK so this is easy
    return new PartReaderStd();
}
//---------- Protected Methods ---------------------------
// RECURSIVE
protected String listCell(Cell cell, int level) {

    String text = ""; 
    Enumeration names = cell.enumerateNodeNames();
    while (names.hasMoreElements()) {
        String name = (String)names.nextElement();
        // Build prefix of spaces, which is level * indent
        // Why doesn't Java have an easy way to do this?
        int indent = 4;
        int spaces = level * indent;
        String prefix = "";
        for (int j = 1; j <= spaces; j++) {
            prefix += " ";
        }
        // Build line
        text += prefix + name + "\n";
        
        // RECURSE if container
        Node node = cell.getNode(name);
        if (node.isContainer()) {
            text += listCell((Cell)node.getPart(), level + 1);            
        }
    }
    return text;
}
//---------- Private Methods -----------------------------
//--- Std
private static void print(String text) {
    System.out.println("PartReaderStd" + text);
}


} // End class