package org.jcon.core.part.std;

import org.jcon.core.part.Node;
import org.jcon.core.part.NodeUser;

/**
* This class represents a single node in a hierarchial
* System Tree. Each node has one part. A node can be a 
* container or leaf. Containers hold zero or more nodes.
* The terms part and node are fairly interchangable.
* <p>
* The name "MyCell" is not allowed. This is reserved for use
* in Cell.findPart(String name).
* <p>
* Responsibilites are to maintain the state that applies
* to all parts, such as name, parent and isCore. The design
* philosophy is it should remain small and simple, serving
* only an "identification and properties role", with almost no 
* behavior.
* <p>
* The boolean properties are all designed with the most 
* common case as false, which is the default. For example
* we use "isHidden" instead of "isVisible".
*
* @author Jack Harich
*/
public class NodeStd implements Node {

//---------- Protected Fields ------------------------------------
protected String name;
protected Object part;
protected Node   parentNode;

// The default on all of these is documented as false
protected boolean isCore;
protected boolean isContainer;
protected boolean isCompanion;
protected boolean isInheritable;
protected boolean isVirtual;
protected boolean isHidden;

//---------- Node Implementation ---------------------------------
//----- name
/**
* Sets the node or part name. This must be unique per container.
* It must be alphanumeric, may contain periods, and must
* not contain blanks or slashes. It cannot be null or a
* zero length String. This is a required property.
* <p>
* Our convention is to capatalize the first letter of each
* word making up the self-documenting part name. Examples
* are "Root", "ControlRegistry" and "UserSecurity".
* <p>
* If part names are used for finding the part in other
* containers, then to avoid name space conflicts the names
* need careful design. This can achieve branch or system
* uniqueness. This will happen if service parts are used
* a lot. To support unique naming we allow periods in the
* name, such as "org.jcon.security.UserAuthorization".
* @param name  the name of the node.
*/
public void setName(String name) {
    // TODO - Validation
    this.name = name;
}
/*
* Returns the part name.
* @return  the name of the part.
*/
public String getName() {
    return name;
}
//----- part
/**
* Sets the node's part, which may be any object whatsoever.
* The object should be unique per container and probably
* unique per System Tree. This is a required property.
* <p>
* If the part implements NodeUser then setNode(Node) will
* be called, using "this" node.
* @param part  the node's part.
*/
public void setPart(Object part) {
    this.part = part;
    
    if (part instanceof NodeUser) {    
        ((NodeUser)part).setNode(this);
    }
}
/**
* Returns the node's part.
* @return  the node's part.
*/
public Object getPart() {
    return part;
}
//----- parentNode 
/*
* Sets the node's parent node. This is required for 
* all nodes in a tree except the root, which is null.
* @param parentNode  the parent of this node or null if
*                    this is the root node.
*/
public void setParentNode(Node parentNode) {
    this.parentNode = parentNode;
}
/*
* Returns the node's parent node or null if this is
* the root.
* @return  the parent of this node or null if this is the root node.
*/
public Node getParentNode() {
    return parentNode;
}
//----- isCore 
/**
* Sets whether this is a core part or not. The default is
* false.
* @param isCore  true if this is a core part or false if not.
*/
public void setCore(boolean isCore) {
    this.isCore = isCore;
}
/**
* Returns true if this is a core part or false if not.
* <p>
* Parts are divided into two groups:
* core and domain. Core parts are required for a System 
* Tree to run. They are domain neutral, and are normally 
* hidden. Domain parts are the ones supplied by the system
* builder, have  the domain functionality, and form the 
* bulk of a system. The use of core parts by domain parts
* should be minimized and fascaded to preserve core encapsulation.
* @return  true if this is a core part or false if not.
*/
public boolean isCore() {
    return isCore;
}
//----- isContainer 
/**
* Sets whether this is a container part or not. The default
* is false.
* @param isContainer  true if this is a container or false if not.
*/
public void setContainer(boolean isContainer) {
    this.isContainer = isContainer;
}
/**
* Returns true if this is a container part or false if not.
* <p>
* A unique type of core part
* is a  container. It holds other container or leaf parts.
* All other types of parts are leaf parts. A container part
* requires the other core parts to run. Thus the core parts
* form the "container" portion of a system. We do not call
* this "isCell" to allow other types of containers, such as
* organs.
* @return  true if this is a container or false if not.
*/
public boolean isContainer() {
    return isContainer;
}
//----- isCompanion 
/**
* Sets whether this is a companion part or not. The default
* is false.
* @param isCompanion  true if this is a companion or false if not.
*/
public void setCompanion(boolean isCompanion) {
    this.isCompanion = isCompanion;
}
/**
* Returns true if this is a companion part or false if not.
* <p>
* A container may
* have one leaf part that's psychologically associated with
* the container. An example is a registry. In the visual 
* System Tree companions are shown as with their containers
* instead of being a child. This helps conceptual design.
* @return  true if this is a companion or false if not.
*/
public boolean isCompanion() {
    return isCompanion;
}
//----- isInheritable
/**
* Sets whether this is inheritable or not. The default is
* false.
* @param isInheritable  true if this is inheritable or false if not.
*/
public void setInheritable(boolean isInheritable) {
    this.isInheritable = isInheritable;
}
/**
* Returns true if this is inheritable or false if not.
* <p>
* Parts may play a "local"
* or "inheritable" role in their collaborations with other
* parts. Most parts are local, and are used via Messages or 
* occasionally directly. Some parts play a "service" role
* by sitting near the upper region of a System Tree and 
* serving the needs of many parts below by being inheritable.
* This is the Service Architecture Pattern, though that
* pattern is not limited to trees.
* <p>
* We recommend the use of "message services" instead of "part 
* services" for most use. This has looser coupling.
* <p>
* An example of the use of isInheritable is core service
* parts. This allows cores that don't use Messages,
* gives greater speed, and allows part sharing of stateless
* part by many containers for efficiency. One purpose
* of this property is so the designer can declare that a
* part NOT be used as a service, which can improve cell
* encapsulation. Another advantage is automatic management
* of service parts by a service registry or such.
* @return  true if this is inheritable or false if not.
*/
public boolean isInheritable() {
    return isInheritable;
}
//----- isVirtual 
/**
* Sets whether this is a virtual part or not. The default
* is false.
* @param isVirtual  true if this is virtual or false if not.
*/
public void setVirtual(boolean isVirtual) {
    this.isVirtual = isVirtual;
}
/**
* Returns true if this is a virtual part or false if not.
* <p>
* Parts may be normal or 
* "virtual". Normally all parts are defined in Container 
* DK. But virtual parts are not defined by Container DK. 
* This  allows more dynamic system behavior, less DK, and 
* lighter system weight. An example of virtual parts is 
* core parts in subcells below the root. 
* @return  true if this is virtual or false if not.
*/
public boolean isVirtual() {
    return isVirtual;
}
//----- isHidden
/**
* Sets whether this is a hidden part or not. The default
* is false.
* @param isHidden  true if this is hidden or false if not.
*/
public void setHidden(boolean isHidden) {
    this.isHidden = isHidden;
}
/**
* Returns true if this is a hidden part or false if not.
* <p>
* Most parts are visible in the
* visual System Tree tool and to "get all parts in a 
* container" calls, etc. But the core parts (except for 
* container) are hidden, because that would clutter up 
* things visually, and expose them to possible misuse by
* non-core parts. This also helps preserve core 
* encapsulation.
* <p>
* This property can be used to control the behavior of
* classes like "PartReader" to hide sensitive parts.
* (It's a little muddy because of the show container
* exception)
* @return  true if this is hidden or false if not.
*/
public boolean isHidden() {
    return isHidden;
}
//----- Other
/**
* Returns the root node, which has no parent node.
* @return  the root node.
*/
public Node getRootNode() {
    if (parentNode == null) {
        return this;
    } else {
        return parentNode.getRootNode();
    }
}
/**
* Returns a String representation of the node. The standard
* is the name of the node.
* @return  a String representation of the node.
*/
public String toString() {
    return name;
}


} // End class