package org.jcon.core.part.std;

import org.jcon.core.part.Cell;
import org.jcon.core.part.Node;
import org.jcon.core.part.PartUser;
import org.jcon.core.tron.PropMap;
import org.jcon.core.tron.std.PropMapStd;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

/**
* This class represents a cell in a living organism,
* which in UHR is a container in a System Tree. A Cell
* holds zero or more Nodes, each of which may be a Cell
* or leaf.
* <p>
* Responsiblities are to manage the collection of nodes
* with the option of events. The nodes are maintained in
* the order added, allowing the designer to better
* organize the System Tree.
* <p>
* Unfortunately this class has the additional responsibility
* of handling the PartUser optional interface. This may be
* done differently in different cores, especially if a
* cell lifecycle part is present.
*
* @author Jack Harich
*/

public class CellStd implements Cell {

//---------- Protected Fields ------------------------------------
protected PropMap nodes = new PropMapStd(false, true, null);
protected Node    myNode;

// Key = partName, Object = Vector of PartUsers
protected Hashtable partNeeders = new Hashtable();

//---------- Cell Implementation ---------------------------------
/**
* Appends the node after the last index. Throws an
* IllegalStateException if the node has a duplicate name.
* @param node  the node to append.
*/
public void appendNode(Node node) {
    duplicateNameCheck(node);
    nodes.put(node.getName(), node);
    initPart(node.getPart(), node.getName());    
}
/**
* Inserts the node before the nodeBefore, which must be in
* the Cell. Throws an IllegalStateException if the nodeBefore
* is not found or the node has a duplicate name.
* @param node        the node to insert.
* @param nodeBefore  the node to insert before.
*/
public void insertNodeBefore(Node node, Node nodeBefore) {
    duplicateNameCheck(node);
    nodes.insertBefore(nodeBefore.getName(), node.getName(), node); 
    initPart(node.getPart(), node.getName());
}
/**
* Removes all nodes from the Cell.
*/
public void removeAll() {
    nodes.removeAll();
}
/**
* Removes the node. Returns the node removed or null if none.
* @param node  the node to remove.
*/     
public Node removeNode(Node node) {
    return (Node)nodes.removeKey(node.getName());
}
/**
* Returns the number of nodes in the cell. We use the
* Bean Spec method name style rather than size().
* @return  the number of nodes in the cell.
*/
public int getNodeCount() {
    return nodes.getSize();
}
/**
* Returns an Enumeration of all node names in the cell, in order.
* @return  the node names in thr cell.
*/
public Enumeration enumerateNodeNames() {
    return nodes.enumerateKeys();
}
/**
* Returns the named node in this cell or null if not found.
* @param nodeName  the name of the node to retrieve.
* @return  the node with the nodeName or null if not found..
*/
public Node getNode(String name) {
    return (Node)nodes.get(name);
}
/**
* Returns the node this cell is in.
* @return  the node this cell is in.
*/
public Node getCellNode() {
    return myNode;
}
/**
* Returns the named part in the cell.
* See <a href="org.jcon.core.part.PartReader.html#getPart(java.lang.String)">full documentation</a>.
*
* @param partName  the name of the part to retrieve.
* @return  the part with the partName or null if not found.
*/
public Object getPart(String nodeName) {
    return ((Node)nodes.get(nodeName)).getPart();
}
/**
* Finds and returns the named part. 
* See <a href="org.jcon.core.part.PartReader.html#findPart(java.lang.String)">full documentation</a>.
*
* @param partName  the name of the part to find.
* @return          the part with the partName.
*/
public Object findPart(String partName) {
    // Special case 
    if (partName.equals("MyCell")) return this;
    
    // Perform 4 level search
    Node node = getNode(partName);
    
    // Case 1 - Found in this container
    if (node != null) return node.getPart();
    
    // Search upward in each cell until root encountered
    Node upwardNode = myNode.getParentNode();
    while (upwardNode != null) {
        Cell upwardCell = (Cell)upwardNode.getPart();
        node = upwardCell.getNode(partName);
        if (node != null) break;
        upwardNode = upwardNode.getParentNode();
    }
    // Case 2 - Never found upwardly
    if (node == null) {
        throw new IllegalStateException("Part '" + partName +
        "' not found in this container or above.");
    }
    // Case 3 - Found upwardly, is inheritable part
    if (node.isInheritable()) {
        return node.getPart();
    }
    // Case 4 - Found upwardly, not inheritable part
    throw new IllegalStateException("Part '" + partName +
    "' found upwardly but is not an inheritable part.");
}
//----- NodeUser Implementation
/**
* Sets the node the part is in.
*/
public void setNode(Node node) {
    myNode = node;
}
//----- Replicatable Implementation 
/**
* Returns a copy of itself without mission state.
*/
public Object replicate() {
    // No DK so this is easy
    return new CellStd();
}
//---------- Protected Methods -----------------------------------
protected void duplicateNameCheck(Node node) {

    if (node.getName() == null) {
        throw new IllegalStateException("Cannot handle"
        + " nodes with a null name.");
    }

    if (nodes.containsKey(node.getName())) {
        throw new IllegalStateException("The node named '"
        + node.getName() + "' is already in this cell.");
    }
}
/**
* Handles the PartUser optional interface. This is done in
* a manner that allows parts to be added in any order.
* If a PartUser's needs are never met, no exception or
* such is thrown, and the part will probably error out
* when it needs the missing part. This can be corrected
* in a cell lifecycle approach or by cell DK analysis.
*/
protected void initPart(Object part, String partName) {
    // If PartUser, provide parts needed
    if (part instanceof PartUser) {
        PartUser partUser = (PartUser)part;
        String[] names = partUser.getPartNamesNeeded();
        
        // Supply part for each need
        for (int i = 0; i < names.length; i++) {
            supplyPart(names[i], partUser);
        }    
    }    
    // Check partNeeders for part arrival
    Vector needers = (Vector)partNeeders.get(partName);
    if (needers == null) return; // None awaiting this part
    
    // Part has arrived and has needers
    for (int i = 0; i < needers.size(); i++) {
        PartUser partUser = (PartUser)needers.elementAt(i);
        partUser.setPart(partName, part);
    }
    partNeeders.remove(partName);
}
// Supplies named part to user. If not found, adds user to
// partNeeders for partName.
protected void supplyPart(String partName, PartUser user) {
    Object partNeeded;
    try {
        partNeeded = findPart(partName);
        user.setPart(partName, partNeeded);
        
    } catch(Exception ex) {
        // Not found so add to partNeeders
        Vector needers = (Vector)partNeeders.get(partName);
        if (needers == null) {
            needers = new Vector();
            partNeeders.put(partName, needers);
        }
        needers.addElement(user);
    } 
}
//---------- Private Methods -------------------------------------
//--- Std
private static void print(String text) {
    System.out.println("CellStd" + text);
}


} // End class