package org.jcon.core.part.std;

import org.jcon.core.part.Cell;
import org.jcon.core.part.CellNodeReplicator;
import org.jcon.core.part.Node;
import org.jcon.core.part.PartReader;
import org.jcon.core.part.PartUser;
import org.jcon.core.part.Replicatable;
import java.util.Enumeration;

/**
* This class is responsible for replicating cell nodes
* given an existing parent cell node to replicate from.
* <p>
* Look closely. When creating new cells, this interface
* adds various core parts using replication. In 
* effect this allows pre-existing life to reproduce and
* create a new life form. This is cellular reproduction.
* <p>
* Note that not all core parts are added, because some are
* inheritable parts. These are available to any cell without
* being in the cell. This greatly reduces cell bloat.
*
* @author Jack Harich
*/

public class CellNodeReplicatorStd implements CellNodeReplicator {

//---------- CellNodeReplicator Implementation -------------------
/**
* Creates a new node with a new cell. The cell contains
* the parts necessary for core work. Note the new node
* does not yet have a name.
* @param fromNode  the node to create a new cell node from.
* @return          the node created.
*/
public Node createCellNode(Node fromNode) {

    if (fromNode == null) throw new IllegalArgumentException(
        "The fromNode is null. Cannot proceed since fromNode is required.");
        
    Cell fromCell = (Cell)fromNode.getPart();
    Cell newCell  = (Cell)fromCell.replicate();
    
    // Populate newCell with core, non-container, 
    //   non-inheritable parts in fromCell
    Enumeration names = fromCell.enumerateNodeNames();
    while (names.hasMoreElements()) {
        String name = (String)names.nextElement();
        Node node = fromCell.getNode(name);
        // Use the carefully designed Node properties
        // We may switch to NideisContainerHelper later if needed
        if (node.isCore() && ! node.isContainer()
                && ! node.isInheritable()) {
            // These MUST all be Replicable
            Replicatable oldPart = (Replicatable)node.getPart();
            Object newPart = oldPart.replicate();
            
            Node newNode = new NodeStd();
            newNode.setName(node.getName());
            newNode.setParentNode(fromNode);
            newNode.setPart(newPart);
            newNode.setCore(true);
            newNode.setVirtual(true);
            newNode.setHidden(true);
            
            newCell.appendNode(newNode);
        }
    }
    // Put newCell in its own node
    // Note we do not set the name
    Node newNode = new NodeStd();
    newNode.setParentNode(fromNode);
    newNode.setPart(newCell);
    newNode.setCore(true);
    newNode.setVirtual(true);
    newNode.setContainer(true);
    
    // Done
    return newNode;
}
//---------- Private Methods -------------------------------------
//--- Std
private static void print(String text) {
    System.out.println("CellNodeReplicatorStd" + text);
}

                        
} // End class                        