package org.jcon.core.part;

import java.util.Enumeration;

/**
* This interface allows reading parts from a cell. It acts
* as a cell Fascade. (More methods will be added)
* <p>
* The PartUser implementation should get its Cell for
* use in getting parts.
*
* @author Jack Harich
*/
// May need to add getParentPart(name)

public interface PartReader extends PartUser, Replicatable {

/**
* Finds and returns the named part or throws an IllegalStateException
* if not found. Can find "local" or "inherited" parts. If
* the part is not found in the cell, an upward
* search is made until the root cell is encountered.
* These are the cases: 
* <p> <p>
* 1. The part is found in the cell and returned. 
* Note it may be a local or inheritable part.
* <p> <p>
* 2. The part is never found, so the exception is thrown.
* <p> <p>
* 3. The part is found upwardly, is an inheritable part and
* is returned.
* <p> <p>
* 4. The part is found upwardly but is not an inheritable
* part, and so the exception is thrown.
* <p>
* A special case is when the cell is needed. Since this 
* usually has a custom name, the reserved partName "MyCell"
* can be used to get the cell the PartReader is in.
* <p>
* A possible feature is if the partName contains slashes, 
* we look downward for it in a standard file path manner.
* We frown on this because it creates tight location dependencies.
*
* @param partName  the name of the part to find.
* @return          the part found.
*/
// Used in Cell, CellStd, PartReaderStd
public Object findPart(String partName);

/**
* Returns the named part in the cell or throws an 
* IllegalStateException if none. <b> Use <code> 
* findPart(String name) </code> instead, except for rare cases of
* restricting the search to the cell. </b> Using findPart allows 
* the designer to move parts upward or downward in the System Tree,
* for greater or lesser reuse via the "inherited part" mechanism,
* as time goes by.
*
* @param name  the name of the part to retrieve.
* @return      the part in the cell with that name.
*/
public Object getPart(String name);

/**
* Returns an Enumeration of the part names in the cell, in order. 
* @return  the part names in the cell.
*/
public Enumeration enumeratePartNames();

/**
* Returns the number of parts in the cell.
* @return  the number of parts in the cell.
*/
public int getPartCount();

/**
* Returns a String representing the System Tree, starting
* at the root or at the cell this PartReader is in.
* Format is one line per part with indentation. This is for 
* simple developement use.
*
* @param startAtRoot  true to start at the root or false to
*                     start at this cell.
*
* @return  a String representing the System Tree.
*/
public String listTree(boolean startAtRoot);


} // End interface