package org.jcon.core.part;

import java.util.Enumeration;

/**
* This interface represents a cell in a living organism,
* which in UHR is a container in a System Tree. A Cell
* holds zero or more Nodes, each of which may be a Cell
* or leaf.
* <p>
* Responsiblities are to manage the collection of nodes
* with the option of events. The nodes are maintained in
* the order added, allowing the designer to better
* organize the System Tree.
*
* @author Jack Harich
*/

public interface Cell extends NodeUser, Replicatable {

/**
* Appends the node after the last index. Throws an
* IllegalStateException if the node has a duplicate name.
* @param node  the node to append.
*/
public void appendNode(Node node);

/**
* Inserts the node before the nodeBefore, which must be in
* the Cell. Throws an IllegalStateException if the nodeBefore
* is not found or the node has a duplicate name.
* @param node        the node to insert.
* @param nodeBefore  the node to insert before.
*/
public void insertNodeBefore(Node node, Node nodeBefore);

/**
* Removes all nodes from the Cell.
*/
public void removeAll();

/**
* Removes the node. Returns the node removed or null if none.
* @param node  the node to remove.
*/     
public Node removeNode(Node node);

/**
* Returns the number of nodes in the cell. We use the
* Bean Spec method name style rather than size().
* @return  the number of nodes in the cell.
*/
public int getNodeCount();

/**
* Returns an Enumeration of all node names in the cell, in order.
* @return  the node names in thr cell.
*/
public Enumeration enumerateNodeNames();

/**
* Returns the named node in this cell or null if not found.
* @param nodeName  the name of the node to retrieve.
* @return  the node with the nodeName or null if not found..
*/
public Node getNode(String nodeName);

/**
* Returns the node this cell is in.
* @return  the node this cell is in.
*/
public Node getCellNode();

/**
* Returns the named part in this cell or null if not found.
* @param partName  the name of the part to retrieve.
* @return  the part with the partName or null if not found.
*/
public Object getPart(String partName);

/**
* Finds and returns the named part. 
* See <a href="org.jcon.core.part.PartReader.html#findPart(java.lang.String)">full documentation</a>.
*
* @param partName  the name of the part to find.
* @return          the part with the partName.
*/
public Object findPart(String partName);

} // End interface